<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\House;
use App\Models\Device;
use App\Models\DeviceControl;

use Illuminate\Http\Request;
class HomeDataController extends Controller
{
   
    public function fullIndex(Request $request)
    {
        $user = $request->user();

        $homes = House::whereHas('users', fn ($q) => $q->where('user_id', $user->id))
            ->with([
                'rooms' => function ($q) {
                    $q->select('id', 'house_id', 'name', 'type', 'floor', 'identifier');
                },
                'rooms.devices' => function ($q) {
                    $q->select('id','name','device_id','secret_key','room_id','house_id','status');
                },
                'rooms.devices.deviceControls.variableType',
                'rooms.devices.deviceControls.widgets',
            ])
            ->get();

        return response()->json($homes);
    }
    
    public function homesWithControls(Request $request)
    {
        $user = $request->user();

        $homes = House::whereHas('users', fn ($q) => $q->where('user_id', $user->id))
            ->with([
                'rooms' => function ($q) {
                    $q->select('id', 'house_id', 'name', 'type', 'floor', 'identifier');
                },
                'rooms.devices' => function ($q) {
                    $q->select('id','name','device_id','secret_key','room_id','house_id','status');
                },
                'rooms.devices.deviceControls.variableType',
                'rooms.devices.deviceControls.widgets',
            ])
            ->get();

        return response()->json($homes);
    }
    
    public function updateDeviceControlValue(Request $request)
    {
        // Validación
        $request->validate([
            'secret_key' => 'required|string',
            'pin' => 'required|string',   // <--- agregamos pin
            'value' => 'required',        // valor a actualizar
        ]);
    
        // Buscar el dispositivo por secret_key
        $device = Device::where('secret_key', $request->secret_key)->first();
    
        if (!$device) {
            return response()->json([
                'message' => 'Secret key no válida o dispositivo no encontrado'
            ], 404);
        }
    
        // Buscar el control dentro del dispositivo que tenga el pin indicado
        $control = DeviceControl::where('device_id', $device->id)
            ->whereJsonContains('sync_with->pin', $request->pin)
            ->first();
    
        if (!$control) {
            return response()->json([
                'message' => 'No se encontró un control con ese pin en este dispositivo'
            ], 404);
        }
    
        // Actualizar valor
        $control->value = $request->value;
        $control->save();
    
        return response()->json([
            'status' => 'ok',
            'device_control_id' => $control->id,
            'new_value' => $control->value,
            'pin' => $request->pin
        ]);
    }


   public function getUserHomes(Request $request)
    {
        $user = $request->user();
        $homes = $user->houses()
            ->select('houses.id', 'houses.name', 'houses.address', 'houses.google_map')
            ->get();
    
        return response()->json([
            'status' => 'ok',
            'homes' => $homes
        ]);
    }
    
    
    
    public function getUserRooms(Request $request)
    {
        $user = $request->user();
    
        // Obtener todas las habitaciones de las casas del usuario
        $rooms = \App\Models\Room::whereHas('house.users', function($q) use ($user) {
            $q->where('user_id', $user->id);
        })
        ->select('id', 'house_id', 'name', 'type', 'description')
        ->with(['house:id,name,address,code']) // opcional: información de la casa
        ->get();
    
        return response()->json([
            'status' => 'ok',
            'rooms' => $rooms
        ]);
    }
    
    
    public function getDeviceControls(Request $request)
{
    $request->validate([
        'secret_key' => 'required|string',
    ]);

    $device = \App\Models\Device::where('secret_key', $request->secret_key)->first();

    if (!$device) {
        return response()->json([
            'status' => 'error',
            'message' => 'Dispositivo no encontrado'
        ], 404);
    }

    // Traer los controles del dispositivo (ajusta los campos segn tu modelo)
    $controls = \App\Models\DeviceControl::where('device_id', $device->id)
        ->get()
        ->map(function ($c) {
            // Suponiendo que en sync_with->pin est el pin
            $pin = null;
            if (isset($c->sync_with) && is_array($c->sync_with) && array_key_exists('pin', $c->sync_with)) {
                $pin = $c->sync_with['pin'];
            } elseif (isset($c->sync_with) && is_string($c->sync_with)) {
                // si lo guardaste como JSON string
                $tmp = json_decode($c->sync_with, true);
                if (isset($tmp['pin'])) $pin = $tmp['pin'];
            }

            return [
                'device_control_id' => $c->id,
                'pin' => (string) $pin,        // pin como string para consistencia
                'value' => (string) $c->value, // value como string "0"/"1"
            ];
        });

    return response()->json([
        'status' => 'ok',
        'device_id' => $device->id,
        'controls' => $controls,
    ]);
}




    
    


}
