<?php

namespace App\Filament\Resources;

use App\Filament\Resources\HouseResource\Pages;
use App\Models\House;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Forms\Components\Placeholder;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Model;
use Filament\Pages\Actions\CreateAction;
use Illuminate\Support\HtmlString;


class HouseResource extends Resource
{
    protected static ?string $model = House::class;
    protected static ?string $navigationGroup = 'Gestionar Hogar';
    protected static ?int $navigationSort = 4;

    protected static ?string $navigationIcon = 'heroicon-o-home-modern';

    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\TextInput::make('name')
                ->label('Nombre')
                ->required(),

            Forms\Components\TextInput::make('cell_phone')
                ->label('Celular')
                ->required(),

            Forms\Components\TextInput::make('address')
                ->label('Dirección')
                ->required(),

            Forms\Components\TextInput::make('google_map')
                ->label('Enlace Google Maps')
                ->required(),

            Forms\Components\TextInput::make('house_numbre')
                ->label('Número de casa'),

            Forms\Components\Hidden::make('created_by')
                ->default(auth()->id()),

            Forms\Components\Select::make('status')
                ->label('Estado del hogar')
                ->options([
                    'active' => 'Activo',
                    'inactive' => 'Inactivo',
                ])
                ->required(),
        ]);
    }

    /**
     * Eager load relaciones importantes para la vista (evita N+1).
     */
    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->with(['creator', 'users', 'rooms']);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('code')
                    ->label('Codigo unico')
                    ->searchable(),

                TextColumn::make('name')
                    ->label('Nombre de la casa')
                    ->searchable(),

                // TextColumn::make('users_list')
                //     ->label('Usuarios')
                //     ->getStateUsing(fn ($record) => $record->users
                //         ->map(fn($u) => ($u->ci ?? '') . ($u->ci ? ' - ' : '') . $u->name . ' (Rol: ' . ($u->pivot->role ?? '-') . ')')
                //         ->join(', ')
                //     )
                //     ->wrap()
                //     ->toggleable(),

                TextColumn::make('cell_phone')
                    ->label('Celular')
                    ->searchable(),

                TextColumn::make('house_numbre')
                    ->label('Número de casa')
                    ->searchable(),

                // // Conteo de ambientes
                // TextColumn::make('rooms_count')
                //     ->label('Cantidad Ambientes')
                //     ->getStateUsing(fn ($record) => $record->rooms->count()),

                // // (Opcional) lista de ambientes
                // TextColumn::make('rooms_list')
                //     ->label('Ambientes (nombres)')
                //     ->getStateUsing(fn ($record) => $record->rooms->pluck('name')->join(', '))
                //     ->wrap()
                //     ->toggleable(),

                TextColumn::make('status')
                    ->label('Estado')
                    ->badge()
                    ->color(fn ($state) => $state === 'active' ? 'success' : 'danger')
                    ->formatStateUsing(fn ($state) => $state === 'active' ? 'Activo' : 'Inactivo'),

                TextColumn::make('creator.name')
                    ->label('Creado por'),
            ])
            ->filters([])
            ->actions([

               Tables\Actions\Action::make('ver_casa')
                    ->label('Ver casa')
                    ->icon('heroicon-o-eye')
                    ->modalHeading(fn (House $record): string => "Casa: {$record->name}")
                    ->modalWidth('lg')
                     ->color('success')
                    ->form([
                        Forms\Components\Section::make('Información de la casa')
                        ->schema([
                            Forms\Components\Placeholder::make('house_info')
                                ->disableLabel()
                               ->content(fn (House $record) => new HtmlString(
                                "🏠 Nombre: {$record->name}<br>📞 Teléfono: {$record->cell_phone}<br>📍 Dirección: {$record->address}"
                                ))
                        ])
                        ->columns(1),

                Forms\Components\Section::make('Cliente')
                    ->schema([
                        Placeholder::make('users')
                            ->disableLabel()
                            ->content(fn (House $record) => new HtmlString(
                                        "<div class='text-sm leading-relaxed whitespace-pre-line col-span-full'>" .
                                        (
                                            $record->users->where('pivot.status', 1)->count() > 0

                                            ? $record->users->map(function ($u) {
                                                $rol = $u->pivot->role ?? '-';
                                                $status = isset($u->pivot->status) ? ($u->pivot->status ? 'Activo' : 'Inactivo') : '-';
                                                return "{$u->name}";
                                            })->join('<br>')
                                            : "No existen clientes relacionados a esta casa"
                                        ) .
                                        "</div>"
                                    ))
                            ])
                        ->columns(1),

                        // Cantidad de ambientes
                        Forms\Components\Section::make('Ambientes')
                            ->schema([
                                Placeholder::make('rooms_info')
                                    ->disableLabel()
                                    ->content(fn (House $record) => new HtmlString(
                                        "<div class='text-sm leading-relaxed whitespace-pre-line col-span-full'>"
                                        . "Cantidad de ambientes: {$record->rooms->count()}<br>"
                                        . (
                                            $record->rooms->count() > 0
                                            ? $record->rooms->map(function ($r) {
                                                $status = $r->status ? 'Activo' : 'Inactivo';
                                                return "- {$r->name}<br>";
                                            })->join("\n")
                                            : "No existen ambientes registrados"
                                        )
                                        . "</div>"
                                    ))
                            ])
                            ->columns(1),
                    ]),



                // Acción: Agregar ambiente (la tuya original)
                Tables\Actions\Action::make('Agregar ambiente')
                    ->label('Agregar ambiente')
                    ->icon('heroicon-o-plus-circle')
                    ->modalHeading('Nuevo ambiente')
                    ->modalSubmitActionLabel('Crear')
                    ->form([
                        Forms\Components\TextInput::make('name')
                            ->label('Nombre del ambiente')
                            ->required(),

                        Forms\Components\Select::make('type')
                            ->label('Tipo')
                            ->options([
                                'living_room' => 'Sala',
                                'kitchen' => 'Cocina',
                                'bedroom' => 'Dormitorio',
                                'bathroom' => 'Baño',
                                'dining_room' => 'Comedor',
                                'office' => 'Oficina',
                                'garage' => 'Garaje',
                                'garden' => 'Jardín',
                                'other' => 'Otro',
                            ])
                            ->required(),

                        // Forms\Components\TextInput::make('identifier')
                        //     ->label('Identificador único')
                        //     ->required(),

                        Forms\Components\TextInput::make('floor')
                            ->label('Piso')
                            ->default(0),

                        Forms\Components\Textarea::make('description')
                            ->label('Descripción')
                            ->nullable(),

                        Forms\Components\Toggle::make('status')
                            ->label('Estado')
                            ->default(true),
                    ])
                    ->action(function (array $data, House $record) {
                        \App\Models\Room::create([
                            'house_id'    => $record->id,
                            'name'        => $data['name'],
                            'type'        => $data['type'],
                            'floor'       => $data['floor'] ?? 0,
                            'description' => $data['description'] ?? null,
                            'status'      => $data['status'] ? 1 : 0,
                        ]);

                        \Filament\Notifications\Notification::make()
                            ->title('Ambiente creado correctamente')
                            ->success()
                            ->send();
                    }),

                // (Puedes agregar EditAction u otras acciones aquí si lo deseas)
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListHouses::route('/'),
            'create' => Pages\CreateHouse::route('/create'),
            'edit' => Pages\EditHouse::route('/{record}/edit'),
        ];
    }

    public static function getPluralModelLabel(): string
    {
        return 'Hogar';
    }

    public static function canViewAny(): bool
    {
        return auth()->user()?->can('hogar.view');
    }

    public static function canCreate(): bool
    {
        return auth()->user()?->can('hogar.create');
    }

    public static function canEdit(Model $record): bool
    {
        return auth()->user()?->can('hogar.update');
    }

    public static function canDelete(Model $record): bool
    {
        return auth()->user()?->can('hogar.delete');
    }
}
