<?php

namespace App\Filament\Resources;

use App\Filament\Resources\DeviceResource\Pages;
use App\Filament\Resources\DeviceResource\RelationManagers;
use App\Models\Device;
use App\Models\DeviceType;
use App\Models\DeviceFamily;
use App\Models\DeviceModel;
use App\Models\House;
use App\Models\Room;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Get;
use App\Models\User;
use Illuminate\Database\Eloquent\Model;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;


class DeviceResource extends Resource
{
    protected static ?string $model = Device::class;

    protected static ?string $navigationIcon = 'heroicon-o-cpu-chip';
    protected static ?string $navigationGroup = 'Dispositivos Registrados';
    protected static ?int $navigationSort = 7;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            Forms\Components\Select::make('house_id')
                ->label('Casa')
                ->options(House::pluck('name', 'id'))
                ->required()
                ->reactive()
                ->afterStateUpdated(fn (callable $set) => $set('room_id', null)),

            Forms\Components\Select::make('room_id')
                ->label('Ambiente')
                ->options(fn (Get $get) =>
                    Room::where('house_id', $get('house_id'))->pluck('name', 'id')
                )
                ->required()
                ->reactive(),

            Forms\Components\TextInput::make('name')
                ->label('Nombre del dispositivo')
                ->required(),

            Select::make('device_type_id')
    ->label('Tipo')
    ->options(DeviceType::pluck('name', 'id'))
    ->reactive()
    ->afterStateUpdated(fn (callable $set) => $set('device_family_id', null)),

Select::make('device_family_id')
    ->label('Familia')
    ->options(fn (Get $get) =>
        DeviceFamily::query()
            ->when($get('device_type_id'), fn ($q, $id) => $q->where('device_type_id', $id))
            ->pluck('name', 'id')
    )
    ->reactive()
    ->preload() // <- importante para edición
    ->afterStateUpdated(fn (callable $set) => $set('device_model_id', null))
    ->required(),

Select::make('device_model_id')
    ->label('Modelo')
    ->options(fn (Get $get) =>
        DeviceModel::query()
            ->when($get('device_family_id'), fn ($q, $id) => $q->where('device_family_id', $id))
            ->pluck('name', 'id')
    )
    ->reactive()
    ->preload() // <- importante para edición
    ->required(),

            Forms\Components\TextInput::make('device_id')
                ->label('ID del dispositivo')
                ->disabled()
                ->dehydrated(false),

            Forms\Components\TextInput::make('secret_key')
                ->label('Clave secreta')
                ->disabled()
                ->dehydrated(false),

            Forms\Components\Toggle::make('status')
                ->label('Activo')
                ->default(true),
            Forms\Components\Hidden::make('created_by')
    ->default(fn ($record) => $record?->created_by ?? auth()->id())


            ]);


    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                //
                Tables\Columns\TextColumn::make('room.house.users.0.name')->label('Cliente'),
                Tables\Columns\TextColumn::make('room.house.name')
                    ->label('Casa asociada')
                    ->sortable()
                    ->searchable(),
                Tables\Columns\TextColumn::make('name')->label('Nombre'),
                // Tables\Columns\TextColumn::make('model.name')->label('Modelo'),
                Tables\Columns\TextColumn::make('device_id')->label('ID del dispositivo')->copyable(),
                Tables\Columns\TextColumn::make('secret_key')->label('Clave secreta')->limit(12)->copyable(),
                // Tables\Columns\IconColumn::make('status')->boolean()->label('Estado'),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListDevices::route('/'),
            'create' => Pages\CreateDevice::route('/create'),
            'edit' => Pages\EditDevice::route('/{record}/edit'),
        ];
    }

    public static function getPluralModelLabel(): string
    {
        return 'Microcontroladores';
    }

    public static function canViewAny(): bool
    {
        return auth()->user()?->can('dispositivo.view');
    }

    public static function canCreate(): bool
    {
        return auth()->user()?->can('dispositivo.create');
    }

    public static function canEdit(Model $record): bool
    {
        return auth()->user()?->can('dispositivo.update');
    }

    public static function canDelete(Model $record): bool
    {
        return auth()->user()?->can('dispositivo.delete');
    }

   public static function getTableQuery(): Builder
    {
        return parent::getEloquentQuery()->where('status', 1);
    }

}

